<?php

use App\Models\Karpet;
use Livewire\Volt\Component;
use Livewire\WithPagination;
use Livewire\WithFileUploads;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;


new class extends Component {
    use WithPagination, WithFileUploads;

    // #[Url(as: 'search')]
    public $search = '';

    // #[Url(as: 'kategori')]
    public $selectedKategori = '';

    // #[Url(as: 'ketebalan')]
    public $selectedKetebalan = '';

    public $showCreateModal = false;
    public $showEditModal = false;
    public $showDeleteModal = false;
    public $selectedKarpet = null;
    public $showImportModal = false;
    public $importFile;


    // Form properties - Updated untuk skema baru
    public $nama = '';
    public $kategori = '';
    public $ketebalan = '';
    public $stok_per_meter = '';
    public $minimum_stok_per_meter = 10;
    public $stok_per_roll = '';
    public $minimum_stok_per_roll = 10;
    public $harga_per_meter = '';
    public $deskripsi = '';
    public $foto;
    public $currentFoto = '';

    protected $rules = [
        'nama' => 'required|string|max:255',
        'kategori' => 'required|string|max:100',
        'ketebalan' => 'required|string|max:50',
        'stok_per_meter' => 'required|integer|min:0',
        'minimum_stok_per_meter' => 'required|integer|min:1',
        'stok_per_roll' => 'required|integer|min:0',
        'minimum_stok_per_roll' => 'required|integer|min:1',
        'harga_per_meter' => 'required|integer|min:0',
        'deskripsi' => 'nullable|string|max:1000',
        'foto' => 'nullable|image|max:2048',
    ];

    // protected $rules += [ // Tambahkan aturan validasi
    //     'importFile' => 'required|file|mimes:csv,txt',
    // ];


    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function updatedSelectedKategori()
    {
        $this->resetPage();
    }

    public function updatedSelectedKetebalan()
    {
        $this->resetPage();
    }

    public function getKarpetsProperty()
    {
        return Karpet::filter($this->search, $this->selectedKategori, $this->selectedKetebalan)
            ->orderBy('created_at', 'desc')
            ->paginate(10);
    }

    public function getKategorisProperty()
    {
        return Karpet::distinct()
            ->pluck('kategori')
            ->filter()
            ->sort()
            ->values()
            ->toArray();
    }

    public function getKetebalansProperty()
    {
        return Karpet::distinct()
            ->pluck('ketebalan')
            ->filter()
            ->sort()
            ->values()
            ->toArray();
    }

    public function openCreateModal()
    {
        $this->resetForm();
        $this->showCreateModal = true;
    }

    public function openEditModal($id)
    {
        $karpet = Karpet::findOrFail($id);
        $this->selectedKarpet = $karpet;
        $this->nama = $karpet->nama;
        $this->kategori = $karpet->kategori;
        $this->ketebalan = $karpet->ketebalan;
        $this->stok_per_meter = $karpet->stok_per_meter;
        $this->minimum_stok_per_meter = $karpet->minimum_stok_per_meter;
        $this->stok_per_roll = $karpet->stok_per_roll;
        $this->minimum_stok_per_roll = $karpet->minimum_stok_per_roll;
        $this->harga_per_meter = $karpet->harga_per_meter;
        $this->deskripsi = $karpet->deskripsi;
        $this->currentFoto = $karpet->foto ?? '';
        $this->foto = null;
        $this->showEditModal = true;
    }

    public function openDeleteModal($id)
    {
        $this->selectedKarpet = Karpet::findOrFail($id);
        $this->showDeleteModal = true;
    }

    public function closeModal()
    {
        $this->showCreateModal = false;
        $this->showEditModal = false;
        $this->showDeleteModal = false;
        $this->selectedKarpet = null;
        $this->resetForm();
        $this->resetValidation();
    }

    public function resetForm()
    {
        $this->nama = '';
        $this->kategori = '';
        $this->ketebalan = '';
        $this->stok_per_meter = '';
        $this->minimum_stok_per_meter = 10;
        $this->stok_per_roll = '';
        $this->minimum_stok_per_roll = 10;
        $this->harga_per_meter = '';
        $this->deskripsi = '';
        $this->foto = null;
        $this->currentFoto = '';
    }

    public function store()
    {
        $this->validate();

        // Upload foto ke public directory
        $fotoPath = null;
        if ($this->foto) {
            // Buat folder jika belum ada
            $publicPath = public_path('images/karpets');
            if (!file_exists($publicPath)) {
                mkdir($publicPath, 0755, true);
            }

            // Generate nama file unik
            $filename = time() . '_' . uniqid() . '.' . $this->foto->getClientOriginalExtension();
            $fullPath = $publicPath . '/' . $filename;

            // Copy file dari temporary ke public
            if (copy($this->foto->getRealPath(), $fullPath)) {
                chmod($fullPath, 0644);
                $fotoPath = 'images/karpets/' . $filename;
            }
        }

        Karpet::create([
            'nama' => $this->nama,
            'kategori' => $this->kategori,
            'ketebalan' => $this->ketebalan,
            'stok_per_meter' => $this->stok_per_meter,
            'minimum_stok_per_meter' => $this->minimum_stok_per_meter,
            'stok_per_roll' => $this->stok_per_roll,
            'minimum_stok_per_roll' => $this->minimum_stok_per_roll,
            'harga_per_meter' => $this->harga_per_meter,
            'deskripsi' => $this->deskripsi,
            'foto' => $fotoPath,
        ]);

        $this->closeModal();
        session()->flash('success', 'Karpet berhasil ditambahkan!');
    }

    public function update()
    {
        $this->validate();

        // Update foto
        $fotoPath = $this->currentFoto;

        if ($this->foto) {
            // Hapus foto lama jika ada
            if ($this->currentFoto && file_exists(public_path($this->currentFoto))) {
                unlink(public_path($this->currentFoto));
            }

            // Buat folder jika belum ada
            $publicPath = public_path('images/karpets');
            if (!file_exists($publicPath)) {
                mkdir($publicPath, 0755, true);
            }

            // Generate nama file unik
            $filename = time() . '_' . uniqid() . '.' . $this->foto->getClientOriginalExtension();
            $fullPath = $publicPath . '/' . $filename;

            // Copy file dari temporary ke public
            if (copy($this->foto->getRealPath(), $fullPath)) {
                chmod($fullPath, 0644);
                $fotoPath = 'images/karpets/' . $filename;
            }
        }

        $this->selectedKarpet->update([
            'nama' => $this->nama,
            'kategori' => $this->kategori,
            'ketebalan' => $this->ketebalan,
            'stok_per_meter' => $this->stok_per_meter,
            'minimum_stok_per_meter' => $this->minimum_stok_per_meter,
            'stok_per_roll' => $this->stok_per_roll,
            'minimum_stok_per_roll' => $this->minimum_stok_per_roll,
            'harga_per_meter' => $this->harga_per_meter,
            'deskripsi' => $this->deskripsi,
            'foto' => $fotoPath,
        ]);

        $this->closeModal();
        session()->flash('success', 'Karpet berhasil diperbarui!');
    }

    public function delete()
    {
        // Hapus foto dari public directory
        if ($this->selectedKarpet->foto && file_exists(public_path($this->selectedKarpet->foto))) {
            unlink(public_path($this->selectedKarpet->foto));
        }

        $this->selectedKarpet->delete();
        $this->closeModal();
        session()->flash('success', 'Karpet berhasil dihapus!');
    }

     public function exportExcel()
    {
        $karpets = Karpet::filter($this->search, $this->selectedKategori, $this->selectedKetebalan)
            ->get();

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Set header
        $sheet->setCellValue('A1', 'ID');
        $sheet->setCellValue('B1', 'Nama Karpet');
        $sheet->setCellValue('C1', 'Kategori');
        $sheet->setCellValue('D1', 'Ketebalan');
        $sheet->setCellValue('E1', 'Stok per Meter');
        $sheet->setCellValue('F1', 'Min Stok per Meter');
        $sheet->setCellValue('G1', 'Stok per Roll');
        $sheet->setCellValue('H1', 'Min Stok per Roll');
        $sheet->setCellValue('I1', 'Harga per Meter');
        $sheet->setCellValue('J1', 'Status Stok');
        $sheet->setCellValue('K1', 'Dibuat Tanggal');

        // Isi data
        $row = 2;
        foreach ($karpets as $karpet) {
            $sheet->setCellValue('A'.$row, $karpet->id);
            $sheet->setCellValue('B'.$row, $karpet->nama);
            $sheet->setCellValue('C'.$row, $karpet->kategori);
            $sheet->setCellValue('D'.$row, $karpet->ketebalan);
            $sheet->setCellValue('E'.$row, $karpet->stok_per_meter);
            $sheet->setCellValue('F'.$row, $karpet->minimum_stok_per_meter);
            $sheet->setCellValue('G'.$row, $karpet->stok_per_roll);
            $sheet->setCellValue('H'.$row, $karpet->minimum_stok_per_roll);
            $sheet->setCellValue('I'.$row, $karpet->harga_per_meter);
            $sheet->setCellValue('J'.$row, $karpet->is_stok_menipis ? 'Menipis' : 'Normal');
            $sheet->setCellValue('K'.$row, $karpet->created_at->format('Y-m-d H:i:s'));
            $row++;
        }

        $writer = new Xlsx($spreadsheet);
        $fileName = 'karpet_export_' . now()->format('Y-m-d_H-i-s') . '.xlsx';
        
        // Simpan sementara
        $tempPath = tempnam(sys_get_temp_dir(), 'karpet_export') . '.xlsx';
        $writer->save($tempPath);

        return response()->download($tempPath, $fileName)->deleteFileAfterSend(true);
    }

    public function importStok()
    {
        $this->validate([
            'importFile' => 'required|file|mimes:xlsx,xls,csv,txt',
        ]);

        $path = $this->importFile->getRealPath();
        $extension = $this->importFile->getClientOriginalExtension();
        
        if (in_array($extension, ['xlsx', 'xls'])) {
            $reader = IOFactory::createReader(ucfirst($extension));
            $spreadsheet = $reader->load($path);
            $sheet = $spreadsheet->getActiveSheet();
            $rows = $sheet->toArray();
        } else {
            // Untuk CSV/TXT
            $file = fopen($path, 'r');
            $rows = [];
            while (($row = fgetcsv($file)) !== false) {
                $rows[] = $row;
            }
            fclose($file);
        }

        // Lewati header
        array_shift($rows);

        $updatedCount = 0;
        $errors = [];

        foreach ($rows as $row) {
            try {
                // Pastikan minimal ada 3 kolom
                if (count($row) < 3) {
                    $errors[] = "Baris tidak valid: " . implode(',', $row) . " - Format baris harus: ID,StokMeter,StokRoll";
                    continue;
                }

                $id = trim($row[0]);
                $stokMeter = isset($row[1]) && trim($row[1]) !== '' ? (int)trim($row[1]) : null;
                $stokRoll = isset($row[2]) && trim($row[2]) !== '' ? (int)trim($row[2]) : null;

                if (!$stokMeter && !$stokRoll) continue;

                $karpet = Karpet::find($id);
                
                if (!$karpet) {
                    $errors[] = "ID {$id} tidak ditemukan";
                    continue;
                }

                $updateData = [];
                if ($stokMeter !== null) $updateData['stok_per_meter'] = $stokMeter;
                if ($stokRoll !== null) $updateData['stok_per_roll'] = $stokRoll;

                $karpet->update($updateData);
                $updatedCount++;

            } catch (\Exception $e) {
                $errors[] = "Error pada baris: " . implode(',', $row) . " - " . $e->getMessage();
            }
        }

        $this->importFile = null;
        $this->showImportModal = false;

        $message = "Berhasil update {$updatedCount} stok karpet!";
        if (!empty($errors)) {
            $message .= " Terdapat " . count($errors) . " error.";
            session()->flash('import_errors', $errors);
        }

        session()->flash('success', $message);
    }



    public function pesanWhatsApp($karpetId)
    {
        $karpet = Karpet::findOrFail($karpetId);
        $message = "Halo, saya tertarik dengan karpet {$karpet->nama} (Kategori: {$karpet->kategori}, Ketebalan: {$karpet->ketebalan}, Harga: {$karpet->harga_formatted}). Apakah masih tersedia?";
        $whatsappUrl = "https://wa.me/6282135743961?text=" . urlencode($message);

        return redirect()->away($whatsappUrl);
    }

    public function pesanWhatsApp2($karpetId)
    {
        $karpet = Karpet::findOrFail($karpetId);
        $message = "Halo, saya tertarik dengan karpet {$karpet->nama} (Kategori: {$karpet->kategori}, Ketebalan: {$karpet->ketebalan}, Harga: {$karpet->harga_formatted}). Apakah masih tersedia?";
        $whatsappUrl = "https://wa.me/6282135743961?text=" . urlencode($message);

        return redirect()->away($whatsappUrl);
    }
}; ?>

<div>
    <style>
        /* Animasi kelap-kelip untuk stok menipis */
        @keyframes blink {

            0%,
            50% {
                opacity: 1;
            }

            51%,
            100% {
                opacity: 0.3;
            }
        }

        .blink-animation {
            animation: blink 2s infinite;
        }

        .stok-menipis {
            background-color: rgba(239, 68, 68, 0.1);
            border-left: 4px solid #ef4444;
        }
    </style>

    <div class="min-h-screen bg-gray-50 -m-8 text-gray-900">
        {{-- Header --}}
        <div class="bg-green-600 text-white py-6">
            <div class="container mx-auto px-4 flex justify-between items-center">
                <h1 class="text-2xl font-bold">Sistem Informasi Stok Karpet Masjid</h1>
               @auth
                <form action="{{route('logout')}}" method="POST">
                    @csrf
                    <flux:button variant="danger" type="submit">Logout</flux:button>
                </form>
                @endauth
            </div>
        </div>

        <div class="container mx-auto px-4 py-8">
            @if (session()->has('success'))
            <div class="mb-6 bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded">
                {{ session('success') }}
            </div>
            @endif
            @if(session()->has('import_errors'))
            <div class="mb-6 bg-red-50 p-4 border border-red-400 rounded max-h-40 overflow-y-auto">
                <h4 class="font-medium text-red-800">Error saat import:</h4>
                <ul class="mt-2 text-sm text-red-700 list-disc pl-5">
                    @foreach(session('import_errors') as $error)
                    <li>{{ $error }}</li>
                    @endforeach
                </ul>
            </div>
            @endif


            <div class="bg-white rounded-lg shadow p-6 mb-6 text-gray-900">
                <div class="grid grid-cols-1 md:grid-cols-4 gap-4">
                    {{-- Search Field --}}
                    <div>
                        <flux:field>
                            <flux:label class="dark:!text-gray-900">Cari Karpet</flux:label>
                            <flux:input wire:model.live.debounce.300ms="search" placeholder="Nama atau kategori..."
                                class="dark:!text-gray-900" />
                        </flux:field>
                    </div>

                    {{-- Kategori Filter --}}
                    <div class="dark:text-gray-900">
                        <flux:field>
                            <flux:label class="dark:!text-gray-900">Kategori Karpet</flux:label>
                            <flux:select wire:model.live="selectedKategori">
                                <flux:select.option value="">Semua Kategori</flux:select.option>
                                @foreach($this->kategoris as $kategori)
                                <flux:select.option value="{{ $kategori }}">{{ $kategori }}</flux:select.option>
                                @endforeach
                            </flux:select>
                        </flux:field>
                    </div>

                    {{-- Ketebalan Filter --}}
                    <div class="dark:text-gray-900">
                        <flux:field>
                            <flux:label class="dark:!text-gray-900">Ketebalan Karpet</flux:label>
                            <flux:select wire:model.live="selectedKetebalan">
                                <flux:select.option value="">Semua Ketebalan</flux:select.option>
                                @foreach($this->ketebalans as $ketebalanItem)
                                <flux:select.option value="{{ $ketebalanItem }}">{{ $ketebalanItem }}
                                </flux:select.option>
                                @endforeach
                            </flux:select>
                        </flux:field>
                    </div>

                    {{-- Actions --}}
                    <div class="flex items-end gap-2">

                        <flux:button wire:click="exportExcel" variant="outline" class="w-full">
                            Ekspor
                        </flux:button>


                        @auth
                        <flux:button wire:click="$set('showImportModal', true)" variant="outline" class="w-full">
                            Impor Stok
                        </flux:button>


                        <flux:button wire:click="openCreateModal"
                            class="w-full bg-green-600 hover:bg-green-700 text-white border-green-600 hover:border-green-700">
                            Tambah
                        </flux:button>
                        @endauth
                    </div>
                </div>
            </div>

            <div class="bg-white rounded-lg shadow overflow-hidden">
                <div class="overflow-x-auto">
                    <table class="w-full">
                        <thead class="bg-green-600 text-white">
                            <tr>
                                <th class="px-4 py-3 text-left">Foto</th>
                                <th class="px-4 py-3 text-left">Nama Karpet</th>
                                <th class="px-4 py-3 text-left">Kategori</th>
                                <th class="px-4 py-3 text-left">Ketebalan</th>
                                <th class="px-4 py-3 text-center">Stok per Meter</th>
                                <th class="px-4 py-3 text-center">Stok per Roll</th>
                                <th class="px-4 py-3 text-left">Harga/meter</th>
                                <th class="px-4 py-3 text-left">Aksi</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-200 text-gray-900">
                            @forelse($this->karpets as $karpet)
                            <tr class="hover:bg-gray-50 {{ $karpet->is_stok_menipis ? 'stok-menipis' : '' }}">
                                <td class="px-4 py-3">
                                    <img src="{{ $karpet->foto_url }}" alt="{{ $karpet->nama }}"
                                        class="w-16 h-16 object-cover rounded-lg">
                                </td>
                                <td class="px-4 py-3 font-medium">{{ $karpet->nama }}</td>
                                <td class="px-4 py-3">{{ $karpet->kategori }}</td>
                                <td class="px-4 py-3">{{ $karpet->ketebalan }}</td>

                                {{-- Stok per Meter --}}
                                <td class="px-4 py-3 text-center">
                                    <div class="flex flex-col items-center">
                                        <span
                                            class="font-medium {{ $karpet->is_stok_per_meter_menipis ? 'text-red-600 blink-animation' : 'text-gray-900' }}">
                                            {{ $karpet->stok_per_meter }}
                                        </span>
                                        <span class="text-xs text-gray-500">min: {{ $karpet->minimum_stok_per_meter
                                            }}</span>
                                        @if($karpet->is_stok_per_meter_menipis)
                                        <span class="text-red-500 text-xs font-semibold blink-animation">Menipis!</span>
                                        @endif
                                    </div>
                                </td>

                                {{-- Stok per Roll --}}
                                <td class="px-4 py-3 text-center">
                                    <div class="flex flex-col items-center">
                                        <span
                                            class="font-medium {{ $karpet->is_stok_per_roll_menipis ? 'text-red-600 blink-animation' : 'text-gray-900' }}">
                                            {{ $karpet->stok_per_roll }}
                                        </span>
                                        <span class="text-xs text-gray-500">min: {{ $karpet->minimum_stok_per_roll
                                            }}</span>
                                        @if($karpet->is_stok_per_roll_menipis)
                                        <span class="text-red-500 text-xs font-semibold blink-animation">Menipis!</span>
                                        @endif
                                    </div>
                                </td>

                                <td class="px-4 py-3 font-medium">{{ $karpet->harga_formatted }}</td>
                                <td class="px-4 py-3">
                                    @auth
                                    <div class="flex gap-2">
                                        <flux:button wire:click="openEditModal({{ $karpet->id }})" size="sm"
                                            variant="outline">
                                            Edit
                                        </flux:button>
                                        <flux:button wire:click="openDeleteModal({{ $karpet->id }})" size="sm"
                                            variant="danger">
                                            Hapus
                                        </flux:button>
                                    </div>
                                    @else
                                    <flux:button wire:click="pesanWhatsApp({{ $karpet->id }})" size="sm"
                                        class="bg-green-600 hover:bg-green-700 text-white">
                                        Whatsapp 1
                                    </flux:button>
                                    <flux:button wire:click="pesanWhatsApp2({{ $karpet->id }})" size="sm"
                                        class="bg-green-600 hover:bg-green-700 text-white">
                                        Whatsapp 2
                                    </flux:button>
                                    @endauth
                                </td>
                            </tr>
                            @empty
                            <tr>
                                <td colspan="8" class="px-4 py-8 text-center text-gray-500">
                                    Tidak ada data karpet ditemukan
                                </td>
                            </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>

                @if($this->karpets->hasPages())
                <div class="px-4 py-3 border-t">
                    {{ $this->karpets->links() }}
                </div>
                @endif
            </div>
        </div>

        {{-- Create Modal --}}
        <flux:modal wire:model="showCreateModal" class="md:w-96">
            <form wire:submit="store">
                <div class="space-y-6">
                    <flux:heading size="lg">Tambah Karpet Baru</flux:heading>

                    <div class="space-y-4">
                        {{-- Foto Upload Field --}}
                        <flux:field>
                            <flux:label>Foto Karpet</flux:label>
                            <input type="file" wire:model="foto" accept="image/*"
                                class="block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-semibold file:bg-green-50 file:text-green-700 hover:file:bg-green-100">
                            <flux:error name="foto" />
                            {{-- Preview --}}
                            @if($foto)
                            <div class="mt-2">
                                <img src="{{ $foto->temporaryUrl() }}" class="w-32 h-32 object-cover rounded-lg">
                            </div>
                            @endif
                        </flux:field>

                        <flux:field>
                            <flux:label>Nama Karpet</flux:label>
                            <flux:input wire:model="nama" placeholder="Masukkan nama karpet" />
                            <flux:error name="nama" />
                        </flux:field>

                        <flux:field>
                            <flux:label>Kategori</flux:label>
                            <flux:input wire:model="kategori" placeholder="Contoh: Turki, Lokal, Premium" />
                            <flux:error name="kategori" />
                        </flux:field>

                        <flux:field>
                            <flux:label>Ketebalan</flux:label>
                            <flux:input wire:model="ketebalan" placeholder="Contoh: 12 mm" />
                            <flux:error name="ketebalan" />
                        </flux:field>

                        {{-- Stok per Meter --}}
                        <div class="grid grid-cols-2 gap-4">
                            <flux:field>
                                <flux:label>Stok per Meter</flux:label>
                                <flux:input wire:model="stok_per_meter" type="number" min="0" />
                                <flux:error name="stok_per_meter" />
                            </flux:field>

                            <flux:field>
                                <flux:label>Min Stok per Meter</flux:label>
                                <flux:input wire:model="minimum_stok_per_meter" type="number" min="1" />
                                <flux:error name="minimum_stok_per_meter" />
                            </flux:field>
                        </div>

                        {{-- Stok per Roll --}}
                        <div class="grid grid-cols-2 gap-4">
                            <flux:field>
                                <flux:label>Stok per Roll</flux:label>
                                <flux:input wire:model="stok_per_roll" type="number" min="0" />
                                <flux:error name="stok_per_roll" />
                            </flux:field>

                            <flux:field>
                                <flux:label>Min Stok per Roll</flux:label>
                                <flux:input wire:model="minimum_stok_per_roll" type="number" min="1" />
                                <flux:error name="minimum_stok_per_roll" />
                            </flux:field>
                        </div>

                        <flux:field>
                            <flux:label>Harga per Meter (Rp)</flux:label>
                            <flux:input wire:model="harga_per_meter" type="number" min="0" />
                            <flux:error name="harga_per_meter" />
                        </flux:field>

                        <flux:field>
                            <flux:label>Deskripsi</flux:label>
                            <flux:textarea wire:model="deskripsi" placeholder="Deskripsi opsional" rows="3" />
                            <flux:error name="deskripsi" />
                        </flux:field>
                    </div>

                    <div class="flex justify-end gap-2">
                        <flux:button type="button" wire:click="closeModal" variant="ghost">
                            Batal
                        </flux:button>
                        <flux:button type="submit"
                            class="bg-green-600 hover:bg-green-700 text-white border-green-600 hover:border-green-700">
                            Simpan
                        </flux:button>
                    </div>
                </div>
            </form>
        </flux:modal>

        {{-- Edit Modal --}}
        <flux:modal wire:model="showEditModal" class="md:w-96">
            <form wire:submit="update">
                <div class="space-y-6">
                    <flux:heading size="lg">Edit Karpet</flux:heading>

                    <div class="space-y-4">
                        {{-- Foto Upload Field untuk Edit --}}
                        <flux:field>
                            <flux:label>Foto Karpet</flux:label>
                            {{-- Current photo --}}
                            @if($currentFoto)
                            <div class="mb-2">
                                <p class="text-xs text-gray-500">Foto saat ini:</p>
                                <img src="{{ asset($currentFoto) }}" class="w-32 h-32 object-cover rounded-lg">
                            </div>
                            @endif
                            <input type="file" wire:model="foto" accept="image/*"
                                class="block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-semibold file:bg-green-50 file:text-green-700 hover:file:bg-green-100">
                            <flux:error name="foto" />
                            {{-- New preview --}}
                            @if($foto)
                            <div class="mt-2">
                                <p class="text-xs text-gray-500">Preview foto baru:</p>
                                <img src="{{ $foto->temporaryUrl() }}" class="w-32 h-32 object-cover rounded-lg">
                            </div>
                            @endif
                        </flux:field>

                        <flux:field>
                            <flux:label>Nama Karpet</flux:label>
                            <flux:input wire:model="nama" placeholder="Masukkan nama karpet" />
                            <flux:error name="nama" />
                        </flux:field>

                        <flux:field>
                            <flux:label>Kategori</flux:label>
                            <flux:input wire:model="kategori" placeholder="Contoh: Turki, Lokal, Premium" />
                            <flux:error name="kategori" />
                        </flux:field>

                        <flux:field>
                            <flux:label>Ketebalan</flux:label>
                            <flux:input wire:model="ketebalan" placeholder="Contoh: 12 mm" />
                            <flux:error name="ketebalan" />
                        </flux:field>

                        {{-- Stok per Meter --}}
                        <div class="grid grid-cols-2 gap-4">
                            <flux:field>
                                <flux:label>Stok per Meter</flux:label>
                                <flux:input wire:model="stok_per_meter" type="number" min="0" />
                                <flux:error name="stok_per_meter" />
                            </flux:field>

                            <flux:field>
                                <flux:label>Min Stok per Meter</flux:label>
                                <flux:input wire:model="minimum_stok_per_meter" type="number" min="1" />
                                <flux:error name="minimum_stok_per_meter" />
                            </flux:field>
                        </div>

                        {{-- Stok per Roll --}}
                        <div class="grid grid-cols-2 gap-4">
                            <flux:field>
                                <flux:label>Stok per Roll</flux:label>
                                <flux:input wire:model="stok_per_roll" type="number" min="0" />
                                <flux:error name="stok_per_roll" />
                            </flux:field>

                            <flux:field>
                                <flux:label>Min Stok per Roll</flux:label>
                                <flux:input wire:model="minimum_stok_per_roll" type="number" min="1" />
                                <flux:error name="minimum_stok_per_roll" />
                            </flux:field>
                        </div>

                        <flux:field>
                            <flux:label>Harga per Meter (Rp)</flux:label>
                            <flux:input wire:model="harga_per_meter" type="number" min="0" />
                            <flux:error name="harga_per_meter" />
                        </flux:field>

                        <flux:field>
                            <flux:label>Deskripsi</flux:label>
                            <flux:textarea wire:model="deskripsi" placeholder="Deskripsi opsional" rows="3" />
                            <flux:error name="deskripsi" />
                        </flux:field>
                    </div>

                    <div class="flex justify-end gap-2">
                        <flux:button type="button" wire:click="closeModal" variant="ghost">
                            Batal
                        </flux:button>
                        <flux:button type="submit"
                            class="bg-green-600 hover:bg-green-700 text-white border-green-600 hover:border-green-700">
                            Update
                        </flux:button>
                    </div>
                </div>
            </form>
        </flux:modal>

        {{-- Delete Modal --}}
        <flux:modal wire:model="showDeleteModal" class="md:w-96">
            <div class="space-y-6">
                <div>
                    <flux:heading size="lg">Konfirmasi Hapus</flux:heading>
                    <p class="mt-2 text-sm text-gray-600">
                        Apakah Anda yakin ingin menghapus karpet <strong>{{ $selectedKarpet?->nama }}</strong>?
                        Tindakan ini tidak dapat dibatalkan.
                    </p>
                </div>

                <div class="flex justify-end gap-2">
                    <flux:button type="button" wire:click="closeModal" variant="ghost">
                        Batal
                    </flux:button>
                    <flux:button wire:click="delete" variant="danger">
                        Hapus
                    </flux:button>
                </div>
            </div>
        </flux:modal>

        {{-- Import Modal --}}
        <flux:modal wire:model="showImportModal" class="md:w-96">
            <div class="space-y-6">
                        <flux:heading size="lg">Update Stok via Excel/CSV</flux:heading>


                <div class="space-y-4">
                    <div class="bg-blue-50 p-4 rounded-lg">
                        <h3 class="font-medium text-blue-800">Format CSV:</h3>
                                       <p class="text-xs mt-2">
                    <b>Format:</b> Excel (.xlsx, .xls) atau CSV<br>
                    <b>Kolom 1:</b> ID Karpet<br>
                    <b>Kolom 2:</b> Stok per Meter (opsional)<br>
                    <b>Kolom 3:</b> Stok per Roll (opsional)
                </p>

                    </div>

                    <flux:field>
                        <flux:label>File Excel/CSV</flux:label>
                        <input type="file" wire:model="importFile" accept=".xlsx,.xls,.csv,.txt"
                            class="block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100">
                        <flux:error name="importFile" />
                    </flux:field>


                    @if(session()->has('import_errors'))
                    <div class="bg-red-50 p-4 rounded-lg max-h-40 overflow-y-auto">
                        <h4 class="font-medium text-red-800">Error saat import:</h4>
                        <ul class="mt-2 text-sm text-red-700 list-disc pl-5">
                            @foreach(session('import_errors') as $error)
                            <li>{{ $error }}</li>
                            @endforeach
                        </ul>
                    </div>
                    @endif
                </div>

                <div class="flex justify-end gap-2">
                    <flux:button type="button" wire:click="$set('showImportModal', false)" variant="ghost">
                        Batal
                    </flux:button>
                    <flux:button wire:click="importStok"
                        class="bg-blue-600 hover:bg-blue-700 text-white border-blue-600 hover:border-blue-700">
                        Proses
                    </flux:button>
                </div>
            </div>
        </flux:modal>

    </div>
</div>